const $ = (id) => document.getElementById(id);

const params = new URLSearchParams(window.location.search);
const runId = params.get('run');
const repo = params.get('repo');
const url = params.get('url');
const secretName = params.get('secret');

let API_BASE = 'https://api.github.com';
let pat = '';

const STEP_NAMES = {
  'Set up job': 'Setting up runner',
  'Run actions/checkout@v4': 'Checking out code',
  'Clone browser container': 'Cloning browser container',
  'Start browser container': 'Starting browser',
  'Inject session': 'Injecting session cookies',
  'Navigate and capture': 'Navigating and capturing',
  'Install cosign': 'Installing Sigstore tools',
  'Sign with Sigstore': 'Creating attestation',
  'Commit capture': 'Committing to repository',
  'Upload artifacts': 'Uploading artifacts'
};

async function init() {
  $('captureUrl').textContent = url || 'Unknown URL';

  const stored = await chrome.storage.local.get(['pat', 'apiBase']);
  pat = stored.pat;
  if (stored.apiBase) API_BASE = stored.apiBase;

  if (!runId || !repo || !pat) {
    showError('Missing required parameters');
    return;
  }

  poll();
}

async function poll() {
  try {
    const run = await fetchRun();
    updateStatus(run);

    if (run.status === 'completed') {
      if (run.conclusion === 'success') {
        await fetchAndDisplayResult();
        if (secretName) await deleteSecret();
      }
    } else {
      setTimeout(poll, 3000);
    }
  } catch (err) {
    showError(err.message);
  }
}

async function fetchRun() {
  const res = await fetch(`${API_BASE}/repos/${repo}/actions/runs/${runId}`, {
    headers: { Authorization: `Bearer ${pat}`, Accept: 'application/vnd.github+json' }
  });
  if (!res.ok) throw new Error(`Failed to fetch run: ${res.status}`);
  return res.json();
}

async function fetchJobs() {
  const res = await fetch(`${API_BASE}/repos/${repo}/actions/runs/${runId}/jobs`, {
    headers: { Authorization: `Bearer ${pat}`, Accept: 'application/vnd.github+json' }
  });
  if (!res.ok) return { jobs: [] };
  return res.json();
}

function updateStatus(run) {
  const icon = $('statusIcon');
  const title = $('statusTitle');
  const subtitle = $('statusSubtitle');

  icon.className = 'status-icon';

  if (run.status === 'queued') {
    icon.classList.add('pending');
    icon.textContent = '...';
    title.textContent = 'Queued';
    subtitle.textContent = 'Waiting for runner...';
  } else if (run.status === 'in_progress') {
    icon.classList.add('running');
    icon.textContent = '...';
    title.textContent = 'Capturing';
    subtitle.textContent = 'Workflow in progress...';
    fetchJobs().then(updateSteps);
  } else if (run.status === 'completed') {
    if (run.conclusion === 'success') {
      icon.classList.add('success');
      icon.textContent = '!';
      title.textContent = 'Complete';
      subtitle.textContent = 'Capture successful!';
    } else {
      icon.classList.add('failure');
      icon.textContent = 'X';
      title.textContent = 'Failed';
      subtitle.textContent = `Conclusion: ${run.conclusion}`;
    }
    fetchJobs().then(updateSteps);
  }
}

function updateSteps(data) {
  const job = data.jobs?.[0];
  if (!job) return;

  const stepsEl = $('steps');
  stepsEl.innerHTML = job.steps
    .filter(s => STEP_NAMES[s.name])
    .map(s => {
      let iconClass = 'pending';
      let icon = 'o';
      if (s.status === 'completed') {
        iconClass = s.conclusion === 'success' ? 'done' : 'failed';
        icon = s.conclusion === 'success' ? '!' : 'X';
      } else if (s.status === 'in_progress') {
        iconClass = 'running';
        icon = '...';
      }
      return `
        <div class="step">
          <div class="step-icon ${iconClass}">${icon}</div>
          <span class="step-name ${s.status === 'queued' ? 'pending' : ''}">${STEP_NAMES[s.name] || s.name}</span>
        </div>
      `;
    }).join('');
}

async function fetchAndDisplayResult() {
  $('resultLoading').textContent = 'Downloading artifacts';
  $('resultLoading').style.display = 'block';

  const res = await fetch(`${API_BASE}/repos/${repo}/actions/runs/${runId}/artifacts`, {
    headers: { Authorization: `Bearer ${pat}`, Accept: 'application/vnd.github+json' }
  });

  if (!res.ok) {
    showError('Failed to fetch artifacts');
    return;
  }

  const data = await res.json();
  const artifact = data.artifacts?.[0];
  if (!artifact) {
    showError('No artifacts found');
    return;
  }

  $('resultLoading').textContent = 'Extracting files';

  const dlRes = await fetch(`${API_BASE}/repos/${repo}/actions/artifacts/${artifact.id}/zip`, {
    headers: { Authorization: `Bearer ${pat}` }
  });

  if (!dlRes.ok) {
    showError(`Failed to download artifact: ${dlRes.status}`);
    return;
  }

  const zipBlob = await dlRes.blob();

  let files;
  try {
    files = await unzipArtifact(zipBlob);
  } catch (e) {
    showError(`Failed to extract: ${e.message}`);
    return;
  }

  $('resultLoading').style.display = 'none';
  $('result').style.display = 'block';

  // Display screenshot immediately
  if (files['screenshot.png']) {
    const imgUrl = URL.createObjectURL(files['screenshot.png']);
    $('screenshot').src = imgUrl;
  } else {
    $('screenshot').alt = 'Screenshot not found';
  }

  // Bundle view button
  let bundleData = null;
  if (files['bundle.json']) {
    const text = await files['bundle.json'].text();
    bundleData = JSON.parse(text);
    $('viewBundle').onclick = () => {
      const blob = new Blob([JSON.stringify(bundleData, null, 2)], { type: 'application/json' });
      window.open(URL.createObjectURL(blob), '_blank');
    };
    $('viewBundle').style.display = 'inline-block';
  }

  // Parse metadata
  let metadata = null;
  if (files['metadata.json']) {
    metadata = JSON.parse(await files['metadata.json'].text());
    $('resultMeta').textContent = `Captured: ${metadata.timestamp}`;
  }

  // Save to local storage
  await saveCapture({
    runId,
    repo,
    url,
    timestamp: metadata?.timestamp || new Date().toISOString(),
    screenshotBlob: files['screenshot.png'] ? await blobToBase64(files['screenshot.png']) : null,
    bundle: bundleData,
    metadata
  });

  $('savedNotice').style.display = 'block';
}

async function deleteSecret() {
  if (!secretName) return;

  $('cleanupStatus').textContent = 'Removing cookies from GitHub...';
  $('cleanupStatus').style.display = 'block';

  try {
    const res = await fetch(`${API_BASE}/repos/${repo}/actions/secrets/${secretName}`, {
      method: 'DELETE',
      headers: { Authorization: `Bearer ${pat}`, Accept: 'application/vnd.github+json' }
    });

    if (res.ok || res.status === 404) {
      $('cleanupStatus').textContent = 'Cookies removed from GitHub';
      $('cleanupStatus').classList.add('success');
    } else {
      $('cleanupStatus').textContent = `Failed to remove cookies: ${res.status}`;
      $('cleanupStatus').classList.add('error');
    }
  } catch (e) {
    $('cleanupStatus').textContent = `Cleanup error: ${e.message}`;
    $('cleanupStatus').classList.add('error');
  }
}

async function unzipArtifact(blob) {
  const files = {};
  const arrayBuffer = await blob.arrayBuffer();
  const view = new DataView(arrayBuffer);
  const bytes = new Uint8Array(arrayBuffer);

  // Find End of Central Directory (search backwards for signature 0x06054b50)
  let eocdPos = -1;
  for (let i = bytes.length - 22; i >= 0; i--) {
    if (view.getUint32(i, true) === 0x06054b50) {
      eocdPos = i;
      break;
    }
  }
  if (eocdPos < 0) throw new Error('Invalid ZIP: no EOCD');

  const cdOffset = view.getUint32(eocdPos + 16, true);
  const cdEntries = view.getUint16(eocdPos + 10, true);

  // Read Central Directory entries
  let pos = cdOffset;
  for (let i = 0; i < cdEntries; i++) {
    if (view.getUint32(pos, true) !== 0x02014b50) break;

    const method = view.getUint16(pos + 10, true);
    const compSize = view.getUint32(pos + 20, true);
    const nameLen = view.getUint16(pos + 28, true);
    const extraLen = view.getUint16(pos + 30, true);
    const commentLen = view.getUint16(pos + 32, true);
    const localHeaderOffset = view.getUint32(pos + 42, true);

    const fileName = new TextDecoder().decode(bytes.slice(pos + 46, pos + 46 + nameLen));

    // Read local file header to get actual data position
    const localExtraLen = view.getUint16(localHeaderOffset + 28, true);
    const dataStart = localHeaderOffset + 30 + nameLen + localExtraLen;
    const compData = bytes.slice(dataStart, dataStart + compSize);

    if (compSize > 0 && !fileName.endsWith('/')) {
      if (method === 0) {
        files[fileName] = new Blob([compData]);
      } else if (method === 8) {
        try {
          const ds = new DecompressionStream('deflate-raw');
          const writer = ds.writable.getWriter();
          writer.write(compData);
          writer.close();

          const decompressed = await new Response(ds.readable).arrayBuffer();
          files[fileName] = new Blob([decompressed]);
        } catch (e) {
          console.error(`Failed to decompress ${fileName}:`, e);
        }
      }
    }

    pos += 46 + nameLen + extraLen + commentLen;
  }

  return files;
}

async function blobToBase64(blob) {
  return new Promise((resolve) => {
    const reader = new FileReader();
    reader.onloadend = () => resolve(reader.result);
    reader.readAsDataURL(blob);
  });
}

async function saveCapture(capture) {
  const stored = await chrome.storage.local.get(['captures']);
  const captures = stored.captures || [];
  captures.unshift(capture);
  await chrome.storage.local.set({ captures: captures.slice(0, 50) });
}

function showError(msg) {
  $('resultLoading').style.display = 'none';
  $('error').style.display = 'block';
  $('error').textContent = msg;
}

init();
