// GitHub uses libsodium sealed box encryption for secrets
importScripts('lib/libsodium-sumo.js', 'lib/libsodium-wrappers.js');

let sodiumReady = false;

// Initialize sodium
sodium.ready.then(() => {
  sodiumReady = true;
  console.log('[Provenance] libsodium ready');
}).catch(err => {
  console.error('[Provenance] libsodium init failed:', err);
});

// Auto-load config from test bridge if available (for testing)
chrome.runtime.onInstalled.addListener(async () => {
  try {
    const res = await fetch('http://localhost:3000/extension-config');
    if (res.ok) {
      const config = await res.json();
      await chrome.storage.local.set(config);
      console.log('[Provenance] Loaded test config:', Object.keys(config));
    }
  } catch (e) {
    // Not in test environment, that's fine
  }
});

// Poll test bridge for commands (automation)
let polling = false;
async function pollForCommands() {
  if (polling) return;
  polling = true;

  while (true) {
    try {
      const res = await fetch('http://localhost:3000/extension-commands', {
        signal: AbortSignal.timeout(30000)
      });
      if (res.ok) {
        const commands = await res.json();
        for (const cmd of commands) {
          console.log('[Provenance] Received command:', cmd.type);
          const result = await handleCommand(cmd);
          await fetch('http://localhost:3000/extension-response', {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({ id: cmd.id, result })
          });
        }
      }
    } catch (e) {
      // Timeout or not in test env, wait and retry
      await new Promise(r => setTimeout(r, 1000));
    }
  }
}

async function handleCommand(cmd) {
  switch (cmd.type) {
    case 'capture': {
      const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });
      if (!tab) return { error: 'No active tab' };

      const url = new URL(tab.url);
      const domain = url.hostname.replace(/^www\./, '');
      const cookies = await chrome.cookies.getAll({ domain });

      const stored = await chrome.storage.local.get(['repo', 'pat', 'apiBase']);
      if (!stored.repo || !stored.pat) return { error: 'Not configured' };

      const apiBase = stored.apiBase || 'https://api.github.com';
      const secretName = `COOKIES_${domain.replace(/\./g, '_').toUpperCase()}`;

      const session = {
        cookies: cookies.map(c => ({
          name: c.name, value: c.value, domain: c.domain, path: c.path,
          secure: c.secure, httpOnly: c.httpOnly, sameSite: c.sameSite,
          expirationDate: c.expirationDate
        })),
        userAgent: navigator.userAgent
      };

      // Get public key
      const keyRes = await fetch(`${apiBase}/repos/${stored.repo}/actions/secrets/public-key`, {
        headers: { Authorization: `Bearer ${stored.pat}`, Accept: 'application/vnd.github+json' }
      });
      if (!keyRes.ok) return { error: `Failed to get key: ${keyRes.status}` };
      const { key, key_id } = await keyRes.json();

      // Encrypt
      const encrypted = await encryptForGitHub(key, JSON.stringify(session));
      if (encrypted?.error) return { error: encrypted.error };

      // Push secret
      const pushRes = await fetch(`${apiBase}/repos/${stored.repo}/actions/secrets/${secretName}`, {
        method: 'PUT',
        headers: {
          Authorization: `Bearer ${stored.pat}`,
          Accept: 'application/vnd.github+json',
          'Content-Type': 'application/json'
        },
        body: JSON.stringify({ encrypted_value: encrypted, key_id })
      });
      if (!pushRes.ok) {
        const body = await pushRes.text();
        return { error: `Failed to push: ${pushRes.status} ${body}` };
      }

      // Trigger workflow
      const workflowRes = await fetch(`${apiBase}/repos/${stored.repo}/actions/workflows/capture.yml/dispatches`, {
        method: 'POST',
        headers: {
          Authorization: `Bearer ${stored.pat}`,
          Accept: 'application/vnd.github+json',
          'Content-Type': 'application/json'
        },
        body: JSON.stringify({ ref: 'main', inputs: { url: tab.url, secret_name: secretName } })
      });
      if (!workflowRes.ok) {
        const body = await workflowRes.text();
        return { error: `Failed to trigger: ${workflowRes.status} ${body}` };
      }

      // Get the latest run ID
      await new Promise(r => setTimeout(r, 2000));
      const runsRes = await fetch(`${apiBase}/repos/${stored.repo}/actions/runs?per_page=1`, {
        headers: { Authorization: `Bearer ${stored.pat}`, Accept: 'application/vnd.github+json' }
      });
      let runId = null;
      if (runsRes.ok) {
        const runs = await runsRes.json();
        runId = runs.workflow_runs?.[0]?.id;
      }

      // Open status page with secret name for cleanup
      const statusUrl = chrome.runtime.getURL(`status.html?run=${runId}&repo=${encodeURIComponent(stored.repo)}&url=${encodeURIComponent(tab.url)}&secret=${encodeURIComponent(secretName)}`);
      chrome.tabs.create({ url: statusUrl });

      return { success: true, url: tab.url, secretName, cookies: cookies.length, runUrl: statusUrl };
    }

    case 'status': {
      const stored = await chrome.storage.local.get(['repo', 'pat', 'apiBase']);
      return { configured: !!(stored.repo && stored.pat), repo: stored.repo };
    }

    case 'navigate': {
      const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });
      if (tab) {
        await chrome.tabs.update(tab.id, { url: cmd.url });
        // Wait for page to load
        await new Promise(r => setTimeout(r, 3000));
      }
      return { success: true, url: cmd.url };
    }

    default:
      return { error: `Unknown command: ${cmd.type}` };
  }
}

// Start polling after a short delay
setTimeout(pollForCommands, 1000);

// Keep service worker alive with periodic alarm
chrome.alarms.create('keepAlive', { periodInMinutes: 0.5 });
chrome.alarms.onAlarm.addListener((alarm) => {
  if (alarm.name === 'keepAlive') {
    console.log('[Provenance] Keep-alive tick');
  }
});

chrome.runtime.onMessage.addListener((message, sender, sendResponse) => {
  if (message.type === 'encryptSecret') {
    encryptForGitHub(message.publicKey, message.secret)
      .then(sendResponse)
      .catch(err => {
        console.error('[Provenance] Encrypt error:', err);
        sendResponse({ error: err.message });
      });
    return true;
  }
});

async function encryptForGitHub(publicKeyB64, secret) {
  // Wait for sodium to be ready
  let attempts = 0;
  while (!sodiumReady && attempts < 100) {
    await new Promise(r => setTimeout(r, 50));
    attempts++;
  }
  if (!sodiumReady) throw new Error('libsodium not ready');

  console.log('[Provenance] Encrypting with key:', publicKeyB64.slice(0, 20) + '...');

  const publicKey = sodium.from_base64(publicKeyB64, sodium.base64_variants.ORIGINAL);
  const secretBytes = sodium.from_string(secret);

  // Use libsodium's sealed box (crypto_box_seal)
  const encrypted = sodium.crypto_box_seal(secretBytes, publicKey);

  const result = sodium.to_base64(encrypted, sodium.base64_variants.ORIGINAL);
  console.log('[Provenance] Encrypted length:', result.length);
  return result;
}
