const $ = (id) => document.getElementById(id);

// Configurable API base for testing (set via chrome.storage.local.apiBase)
let API_BASE = 'https://api.github.com';

let currentTab = null;
let cookies = [];

async function init() {
  const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });
  currentTab = tab;

  const url = new URL(tab.url);
  $('currentUrl').textContent = url.hostname + url.pathname;

  const domain = url.hostname.replace(/^www\./, '');
  cookies = await chrome.cookies.getAll({ domain });
  $('cookieCount').textContent = `🍪 ${cookies.length} cookies`;

  const stored = await chrome.storage.local.get(['repo', 'pat', 'apiBase']);
  if (stored.repo) $('repo').value = stored.repo;
  if (stored.pat) $('pat').value = stored.pat;
  if (stored.apiBase) API_BASE = stored.apiBase;

  updateButtonState();

  $('viewHistory').addEventListener('click', (e) => {
    e.preventDefault();
    chrome.tabs.create({ url: chrome.runtime.getURL('history.html') });
  });
}

function updateButtonState() {
  const repo = $('repo').value.trim();
  const pat = $('pat').value.trim();
  $('captureBtn').disabled = !repo || !pat || cookies.length === 0;
}

$('repo').addEventListener('input', () => {
  chrome.storage.local.set({ repo: $('repo').value.trim() });
  updateButtonState();
});

$('pat').addEventListener('input', () => {
  chrome.storage.local.set({ pat: $('pat').value.trim() });
  updateButtonState();
});

$('captureBtn').addEventListener('click', async () => {
  const repo = $('repo').value.trim();
  const pat = $('pat').value.trim();
  const url = currentTab.url;

  setStatus('pending', 'Pushing cookies to GitHub secret...');
  $('captureBtn').disabled = true;

  try {
    const domain = new URL(url).hostname.replace(/^www\./, '').replace(/\./g, '_').toUpperCase();
    const secretName = `COOKIES_${domain}`;

    const session = {
      cookies: cookies.map(c => ({
        name: c.name,
        value: c.value,
        domain: c.domain,
        path: c.path,
        secure: c.secure,
        httpOnly: c.httpOnly,
        sameSite: c.sameSite,
        expirationDate: c.expirationDate
      })),
      userAgent: navigator.userAgent
    };

    await pushSecret(repo, pat, secretName, JSON.stringify(session));
    setStatus('pending', 'Triggering capture workflow...');

    const runUrl = await triggerWorkflow(repo, pat, url, secretName);
    setStatus('success', `Workflow started! Opening status page...`);
  } catch (err) {
    setStatus('error', err.message);
  } finally {
    $('captureBtn').disabled = false;
  }
});

function setStatus(type, message) {
  const el = $('status');
  el.className = `status ${type}`;
  el.textContent = message;
}

async function getRepoPublicKey(repo, pat) {
  const res = await fetch(`${API_BASE}/repos/${repo}/actions/secrets/public-key`, {
    headers: { Authorization: `Bearer ${pat}`, Accept: 'application/vnd.github+json' }
  });
  if (!res.ok) throw new Error(`Failed to get repo public key: ${res.status}`);
  return res.json();
}

async function encryptSecret(publicKey, secret) {
  const keyBytes = Uint8Array.from(atob(publicKey), c => c.charCodeAt(0));
  const secretBytes = new TextEncoder().encode(secret);

  const cryptoKey = await crypto.subtle.importKey(
    'raw', keyBytes, { name: 'X25519' }, false, []
  );

  // GitHub uses libsodium sealed box - we need tweetnacl or similar
  // For now, use the simpler approach via service worker
  return chrome.runtime.sendMessage({
    type: 'encryptSecret',
    publicKey,
    secret
  });
}

async function pushSecret(repo, pat, name, value) {
  const { key, key_id } = await getRepoPublicKey(repo, pat);
  const encrypted = await encryptSecret(key, value);

  console.log('[Provenance] Encrypted result:', typeof encrypted, encrypted?.error || encrypted?.slice?.(0, 50));

  if (encrypted?.error) throw new Error(`Encryption failed: ${encrypted.error}`);

  const res = await fetch(`${API_BASE}/repos/${repo}/actions/secrets/${name}`, {
    method: 'PUT',
    headers: {
      Authorization: `Bearer ${pat}`,
      Accept: 'application/vnd.github+json',
      'Content-Type': 'application/json'
    },
    body: JSON.stringify({ encrypted_value: encrypted, key_id })
  });
  if (!res.ok) {
    const body = await res.text();
    console.error('[Provenance] Push failed:', res.status, body);
    throw new Error(`Failed to push secret: ${res.status} - ${body}`);
  }
}

async function triggerWorkflow(repo, pat, url, secretName) {
  const res = await fetch(`${API_BASE}/repos/${repo}/actions/workflows/capture.yml/dispatches`, {
    method: 'POST',
    headers: {
      Authorization: `Bearer ${pat}`,
      Accept: 'application/vnd.github+json',
      'Content-Type': 'application/json'
    },
    body: JSON.stringify({
      ref: 'main',
      inputs: { url, secret_name: secretName }
    })
  });
  if (!res.ok) throw new Error(`Failed to trigger workflow: ${res.status}`);

  // Wait for run to appear then get run ID
  await new Promise(r => setTimeout(r, 2000));
  const runsRes = await fetch(`${API_BASE}/repos/${repo}/actions/runs?per_page=1`, {
    headers: { Authorization: `Bearer ${pat}`, Accept: 'application/vnd.github+json' }
  });

  let runId = null;
  if (runsRes.ok) {
    const runs = await runsRes.json();
    runId = runs.workflow_runs?.[0]?.id;
  }

  // Open status page with secret name for cleanup
  const statusUrl = chrome.runtime.getURL(`status.html?run=${runId}&repo=${encodeURIComponent(repo)}&url=${encodeURIComponent(url)}&secret=${encodeURIComponent(secretName)}`);
  chrome.tabs.create({ url: statusUrl });
  return statusUrl;
}

init();
