const $ = (id) => document.getElementById(id);

let captures = [];
let selectedIndex = -1;

async function init() {
  const stored = await chrome.storage.local.get(['captures']);
  captures = stored.captures || [];

  if (captures.length === 0) {
    $('empty').style.display = 'block';
    return;
  }

  renderCaptures();
}

function renderCaptures() {
  $('captures').innerHTML = captures.map((c, i) => `
    <div class="capture-card" data-index="${i}">
      <img class="capture-thumb" src="${c.screenshotBlob || ''}" alt="">
      <div class="capture-info">
        <div class="capture-url">${c.url}</div>
        <div class="capture-time">${formatTime(c.timestamp)}</div>
      </div>
    </div>
  `).join('');

  document.querySelectorAll('.capture-card').forEach(card => {
    card.addEventListener('click', () => openModal(parseInt(card.dataset.index)));
  });
}

function formatTime(ts) {
  const d = new Date(ts);
  return d.toLocaleString();
}

function openModal(index) {
  selectedIndex = index;
  const c = captures[index];

  $('modalImg').src = c.screenshotBlob || '';
  $('modalUrl').textContent = c.url;
  $('modalTime').textContent = `Captured: ${formatTime(c.timestamp)}`;
  $('modalRepo').textContent = `Repository: ${c.repo}`;

  $('modal').classList.add('open');
}

$('modalClose').addEventListener('click', () => {
  $('modal').classList.remove('open');
});

$('modal').addEventListener('click', (e) => {
  if (e.target === $('modal')) {
    $('modal').classList.remove('open');
  }
});

$('modalBundle').addEventListener('click', () => {
  const c = captures[selectedIndex];
  if (c.bundle) {
    const blob = new Blob([JSON.stringify(c.bundle, null, 2)], { type: 'application/json' });
    window.open(URL.createObjectURL(blob), '_blank');
  }
});

$('modalDelete').addEventListener('click', async () => {
  captures.splice(selectedIndex, 1);
  await chrome.storage.local.set({ captures });
  $('modal').classList.remove('open');

  if (captures.length === 0) {
    $('captures').innerHTML = '';
    $('empty').style.display = 'block';
  } else {
    renderCaptures();
  }
});

document.addEventListener('keydown', (e) => {
  if (e.key === 'Escape') {
    $('modal').classList.remove('open');
  }
});

init();
